/* Contain the entire script within a function because REPORTER only has a single JavaScript realm
 * for the entire session. */
pelvis_score();

/**
 * Performs the pelvis score calculation for the specified models
 */
function pelvis_score() {
    let template = Template.GetCurrent();
    let models = get_model_list();
    let body_region_label = "pelvis";

    let occupants = ["DRIVER"];

    for (let m of models) {
        for (let occ of occupants) {
            LogPrint(`Calculating ${m} ${occ} ${body_region_label} score...`);

            /* Create a status object to track whether REPORTER Variables are all present and valid.
             * <success> is initially true but will be set to false if anything missing or invalid. */
            let status = { success: true, missing: [], invalid: [] };

            let pelvis_force = get_variable_value(status, `${m}_${occ}_PELVIS_FORCE_MAX_VALUE`, "float");
            let lumbar_shear = get_variable_value(status, `${m}_${occ}_LUMBAR_SHEAR_MAX_VALUE`, "float");
            let lumbar_axial = get_variable_value(status, `${m}_${occ}_LUMBAR_AXIAL_MAX_VALUE`, "float");
            let lumbar_torsion = get_variable_value(status, `${m}_${occ}_LUMBAR_TORSION_MAX_VALUE`, "float");
            let PELVIS_PUBIC_SYMPHYSIS = 2.8; //kN
            let PELVIS_LUMBAR_FY = 2; //kN
            let PELVIS_LUMBAR_FZ = 3.5; //kN
            let PELVIS_LUMBAR_MX = 120; //Nm
            let PELVIS_EXCEEDANCE_MODIFIER = -4.0;

            /* Note GetScore() returns Number.NaN if data is "Missing" or null */
            let PELVIS_FORCE_SCORE = GetScore(pelvis_force, PELVIS_PUBIC_SYMPHYSIS, PELVIS_EXCEEDANCE_MODIFIER);
            let LUMBAR_SHEAR_SCORE = GetScore(lumbar_shear, PELVIS_LUMBAR_FY, PELVIS_EXCEEDANCE_MODIFIER);
            let LUMBAR_AXIAL_SCORE = GetScore(lumbar_axial, PELVIS_LUMBAR_FZ, PELVIS_EXCEEDANCE_MODIFIER);
            let LUMBAR_TORSION_SCORE = GetScore(lumbar_torsion, PELVIS_LUMBAR_MX, PELVIS_EXCEEDANCE_MODIFIER);

            let pelvis_and_lumbar_modifier = Math.min(
                PELVIS_FORCE_SCORE,
                LUMBAR_SHEAR_SCORE,
                LUMBAR_AXIAL_SCORE,
                LUMBAR_TORSION_SCORE
            );

            let pelvis_and_lumbar_modifier_str = pelvis_and_lumbar_modifier.toString();

            /** if pelvis_and_lumbar_modifier returned NaN then it means that data is missing for at least one of the values */
            if (Number.isNaN(pelvis_and_lumbar_modifier)) {
                pelvis_and_lumbar_modifier_str = "Missing";
            }

            new Variable(
                template,
                `PELVIS_AND_LUMBAR_MODIFIER`,
                `Result from Automotive Assessments Workflow`,
                pelvis_and_lumbar_modifier_str,
                "General",
                false,
                true
            );

            /** note that if the score is null SetPassFail will return "Fail" */
            PELVIS_FORCE_SCORE = SetPassFailMissing(PELVIS_FORCE_SCORE);
            LUMBAR_SHEAR_SCORE = SetPassFailMissing(LUMBAR_SHEAR_SCORE);
            LUMBAR_AXIAL_SCORE = SetPassFailMissing(LUMBAR_AXIAL_SCORE);
            LUMBAR_TORSION_SCORE = SetPassFailMissing(LUMBAR_TORSION_SCORE);

            new Variable(
                template,
                `${m}_${occ}_PELVIS_FORCE_SCORE`,
                `Result from Automotive Assessments Workflow`,
                PELVIS_FORCE_SCORE,
                "General",
                false,
                true
            );

            new Variable(
                template,
                `${m}_${occ}_LUMBAR_SHEAR_SCORE`,
                `Result from Automotive Assessments Workflow`,
                LUMBAR_SHEAR_SCORE,
                "General",
                false,
                true
            );

            new Variable(
                template,
                `${m}_${occ}_LUMBAR_AXIAL_SCORE`,
                `Result from Automotive Assessments Workflow`,
                LUMBAR_AXIAL_SCORE,
                "General",
                false,
                true
            );

            new Variable(
                template,
                `${m}_${occ}_LUMBAR_TORSION_SCORE`,
                `Result from Automotive Assessments Workflow`,
                LUMBAR_TORSION_SCORE,
                "General",
                false,
                true
            );
        }
    }
}

function GetScore(result, PERFORMANCE_LIMIT, PELVIS_EXCEEDANCE_MODIFIER) {
    if (Number.isNaN(result) || result == null) {
        return Number.NaN;
    }
    if (result > PERFORMANCE_LIMIT) {
        let score = PELVIS_EXCEEDANCE_MODIFIER;
        return score;
    } else {
        return 0;
    }
}

/**
 * If the value is null or not a number return "Missing". Otherwise, return "Pass" if value is 0, else return "Fail".
 * @param {?Number} value
 * @returns {String} "Pass" or "Fail"
 */
function SetPassFailMissing(value) {
    if (value == Number.NaN || value == null) {
        return "Missing";
    } else if (value == 0) {
        return "Pass";
    } else {
        return "Fail";
    }
}
